#!/usr/bin/env python

# Copyright (c) 2018-2020 Intel Corporation
#
# This work is licensed under the terms of the MIT license.
# For a copy, see <https://opensource.org/licenses/MIT>.

"""
Scenarios in which another (opposite) vehicle 'illegally' takes
priority, e.g. by running a red traffic light.
"""

from __future__ import print_function

import py_trees
import carla

from srunner.scenariomanager.carla_data_provider import CarlaDataProvider
from srunner.scenariomanager.scenarioatomics.atomic_behaviors import (ActorDestroy,
                                                                      ActorTransformSetter,
                                                                      AccelerateToCatchUp,
                                                                      SyncArrivalWithAgent,
                                                                      CutIn)
from srunner.scenariomanager.scenarioatomics.atomic_criteria import CollisionTest
from srunner.scenarios.basic_scenario import BasicScenario

from srunner.tools.background_manager import HandleJunctionScenario

from srunner.tools.scenario_helper import generate_target_waypoint

from envs.utils.parse_scenario_config import convert_dict_to_transform

def convert_dict_to_location(actor_dict):
    """
    Convert a JSON string to a Carla.Location
    """
    location = carla.Location(
        x=float(actor_dict['x']),
        y=float(actor_dict['y']),
        z=float(actor_dict['z'])
    )
    return location

class HighwayMerge(BasicScenario):
    """
    """

    def __init__(self, world, ego_vehicles, config, randomize=False, debug_mode=False, criteria_enable=True,
                 timeout=180):
        """
        Setup all relevant parameters and create scenario
        and instantiate scenario manager
        """
        self._world = world
        self._map = CarlaDataProvider.get_map()
        self.timeout = timeout

        self._same_lane_time = 0.3
        self._other_lane_time = 3
        self._change_time = 2
        self._speed_perc = 80
        self._cut_in_distance = 10
        self._extra_space = 170
        self._start_location = convert_dict_to_location(config.other_actors[0].start)

        super().__init__("HighwayMerge",
                         ego_vehicles,
                         config,
                         world,
                         debug_mode,
                         criteria_enable=criteria_enable)

    def _initialize_actors(self, config):
        """
        Custom initialization
        """
        self._other_waypoint = self._map.get_waypoint(self._start_location)
        self._other_transform = self._other_waypoint.transform
        self._cut_in_vehicle = CarlaDataProvider.request_new_actor(
            'vehicle.*', self._other_transform, rolename='scenario',
            attribute_filter={'base_type': 'car', 'has_lights': True}
        )
        self.other_actors.append(self._cut_in_vehicle)
        self._cut_in_vehicle.set_simulate_physics(True)

    def _create_behavior(self):
        behavior = py_trees.composites.Sequence("HighwayExit")
        catch_up = AccelerateToCatchUp(self.other_actors[0],
                                       self.ego_vehicles[1],
                                       trigger_distance=2,
                                       name="CatchUp")
        behavior.add_child(catch_up)
        return behavior

    def _create_test_criteria(self):
        """
        A list of all test criteria will be created that is later used
        in parallel behavior tree.
        """
        criteria = []

        # Add approriate checks for other vehicles
        for vehicle in self.other_actors:
           collision_criterion = CollisionTest(vehicle, name="CollisionTest", terminate_on_failure=True)
           criteria.append(collision_criterion)

        return criteria

    def __del__(self):
        """
        Remove all actors and traffic lights upon deletion
        """
        self.remove_all_actors()
